//
// Maurice Ribble
// 1-22-2005
//
// This is a driver for the Hitachi HM55B Compass Module.  You can buy
// this sensor from www.parallax.com.
//

#define HM55B_IN_OUT PIN_C0
#define HM55B_ENABLE PIN_C1
#define HM55B_CLK    PIN_C2

#define CMD_RESET   0b0000
#define CMD_MEASURE 0b0001
#define CMD_REPORT  0b0011

#define CMD_READY   0b1100

#define NEG_MASK 0b1111100000000000

//
// Helper funtion to get data from the HM55B module.
//
int16 shiftInFromHm55b(int8 numBits)
{
    int8 i;
    int16 out = 0;

    for(i=0; i<numBits; ++i)
    {
        output_high(HM55B_CLK);
        shift_left(&out, 2, input(HM55B_IN_OUT));
        output_low(HM55B_CLK);
    }
    return out;
}

//
// Helper funtion to send data to the HM55B module.
//
void shiftOutToHm55b(int8 var, int8 numBits)
{
    int1 curBit;
    int8 i;

    //
    // We are shifting out the LSB first.
    //
    for(i=0; i<numBits; ++i)
    {
        output_high(HM55B_CLK);
        curBit = shift_right(&var, 1, 0);
        if(curBit)
        {
            output_high(HM55B_IN_OUT);
        }
        else
        {
            output_low(HM55B_IN_OUT);
        }
        output_low(HM55B_CLK);
    }

    output_low(HM55B_IN_OUT);
}

//
// This Hitachi HM55B compass module test program displays x (N/S) and y (W/E)
// axis measurements along with the direction the compass module is pointing,
// measured in degrees clockwise from north.
//
void getCompassXY(int16 &x, int16 &y)
{
    int16 hm55bStatus;

    // Send a reset command to HM55B
    output_high(HM55B_ENABLE);
    output_low(HM55B_ENABLE);
    shiftOutToHm55b(CMD_RESET, 4);

    // Send a start measurement command to HM55B
    output_high(HM55B_ENABLE);
    output_low(HM55B_ENABLE);
    shiftOutToHm55b(CMD_MEASURE, 4);

    while(hm55bStatus != CMD_READY)
    {
        output_high(HM55B_ENABLE);
        output_low(HM55B_ENABLE);
        shiftOutToHm55b(CMD_REPORT, 4);
        hm55bStatus = shiftInFromHm55b(4);
    }

    x = shiftInFromHm55b(11);
    y = shiftInFromHm55b(11);
    output_high(HM55B_ENABLE);

    if(bit_test(x, 10))
    {
        x |= NEG_MASK;
    }

    if(bit_test(y, 10))
    {
        y |= NEG_MASK;
    }

}

float getCompassAngle()
{
    int16 x, y;
    float fx, fy, hypoth, angle;

    getCompassXY(x, y);

    //
    // Now convert this the x and y magnitudes that the HM55B module gives us
    // into something more useful like the angle of deflection from north.
    //

    fx = (signed int16)x;
    fy = (signed int16)y;

    hypoth = sqrt(fx*fx+fy*fy);
    fx /= hypoth;
    fy /= hypoth;
    angle = (atan2(fx, fy))*180/3.14;
    angle -= readEepromFloat(0);
    while(angle < 0)
    {
        angle += 360;
    }
    while(angle > 360)
    {
        angle -= 360;
    }
    return angle;
}

void calibrateCompass()
{
    float offsetAngle;

    //
    // We could get better results if we calculated more angles and then
    // interpolated between them, but I want to keep the calibration simple
    // for now.
    //

    writeEepromFloat(0, 0);  // Need to do this before getCompassAngle()
    offsetAngle = getCompassAngle();
    printf("Offset angle: %f\n", offsetAngle);
    writeEepromFloat(0, offsetAngle);
}


